#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <sys/types.h>    /* \                                 */
#include <sys/socket.h>   /*  > for inet_ntop and its friends. */
#include <arpa/inet.h>    /* /                                 */
#include <netinet/ip.h>   /* to parse IP headers.  */
#include <netinet/tcp.h>  /* to parse TCP headers. */

/**
 * A callback function invoked for each captured packet.
 */
void pcap_packet_cb(u_char* uparam,
	            const struct pcap_pkthdr* hdr,
		    const u_char* p_data)
{
    static int packet_count = 0;
    struct iphdr* ip_hdr;	   /* to get IP protocol data.  */
    struct tcphdr* tcp_hdr;	   /* to get TCP protocol data. */
    char src_ip[100], dst_ip[100];
    int src_port, dst_port;

    packet_count++;

    /* we're only interested in TCP packets. */
    ip_hdr = (struct iphdr*)p_data;  /* the captured data is an IP packet. */
    if (ip_hdr->protocol != IPPROTO_TCP) {
	printf("protocol in IP packet (0x%x) is not TCP\n", ip_hdr->protocol);
	return;
    }

    /* lets get the src and dst addresses - translate from */
    /* network-byte-order binary data. */
    inet_ntop(AF_INET, &ip_hdr->saddr, src_ip, sizeof(src_ip));
    inet_ntop(AF_INET, &ip_hdr->daddr, dst_ip, sizeof(dst_ip));

    /* lets get the port numbers - the payload of the IP packet is TCP.  */
    /* NOTE: in IP, the ihl (IP Header Length) field contains the number */
    /* of 4-octet chunks composing the IP packet's header.               */
    tcp_hdr = (struct tcphdr*)(p_data + ip_hdr->ihl * 4);
    src_port = ntohs(tcp_hdr->source);  /* ports are in network byte order. */
    dst_port = ntohs(tcp_hdr->dest);

    printf("PACKET %d: src %s:%d, dst %s:%d\n",
	   packet_count, src_ip, src_port, dst_ip, dst_port);
}

int main(int argc, char* argv[])
{
    int snap_len = 40;          /* IP hdr (no optoins) + TCP hdr.         */
    int read_timeout_ms = 50;	/* to reduce switching into kernel space. */
    char err_buf[PCAP_ERRBUF_SIZE]; /* here we'll get error strings.      */
    int num_packets_to_capture = 10;
    int rc;

    /* open the device for capturing. */
    pcap_t* pc = pcap_open_live("ppp0",
		    		snap_len,
				0 /*not promiscous*/,
				read_timeout_ms,
				err_buf);
    if (!pc) {
	fprintf(stderr, "Error initializing pcap: %s\n", err_buf);
	exit(1);
    }

    /* capture num_packets_to_capture packets. */
    rc = pcap_loop(pc, num_packets_to_capture, pcap_packet_cb, NULL);
    if (rc == -1)
	pcap_perror(pc, argv[0]);

    /* cleanup - close the device. */
    pcap_close(pc);

    return 0;
}
